<?php

namespace App\Http\Controllers;


use App\Models\Conductores;
use Illuminate\Support\Facades\Request as RequestFacade;
use App\Models\Estudiantes;
use App\Models\Rutas;
use App\Models\SolicitudesRuta;
use App\Models\User;
use App\Models\Vehiculos;
use Inertia\Inertia;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Log;
use App\Mail\SolicitudRutasMail;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

class SolicitudesRutaController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    function __construct()
    {
        $this->middleware('permission:solicitudes_rutas-listar|solicitudes_rutas-crear|solicitudes_rutas-editar|solicitudes_rutas-eliminar', ['only' => ['index']]);
        $this->middleware('permission:solicitudes_rutas-crear', ['only' => ['create', 'store']]);
        $this->middleware('permission:solicitudes_rutas-editar', ['only' => ['edit', 'update']]);
        $this->middleware('permission:solicitudes_rutas-eliminar', ['only' => ['destroy']]);
    }


    //mostrar solicitudes segun id_rutas
    public function show(Request $request, $id_ruta)
    {
        $estado = $request->input('estado', 'pendiente'); // Estado predeterminado: "pendiente"

        $solicitudes = SolicitudesRuta::with(['estudiante' => function ($query) {
            $query->select('id', 'nombre', 'identificacion', 'puntaje', 'telefono', 'foto_perfil');
        }])
            ->where('id_ruta', $id_ruta)
            ->when($estado, function ($query, $estado) {
                $query->where('estado', '=', $estado);
            })
            ->when($request->input('search'), function ($query, $search) {
                $query->whereHas('estudiante', function ($query) use ($search) {
                    $query->where('nombre', 'like', '%' . $search . '%')
                        ->orWhere('identificacion', 'like', '%' . $search . '%');
                });
            })
            ->orderByRaw("FIELD(estado, 'pendiente', 'aprobado', 'rechazado')")
            ->paginate(10)
            ->withQueryString();

        $rutaInfo = Rutas::select('id', 'origen', 'destino', 'puestos_disponibles')->where('id', $id_ruta)->first();

        //return Inertia::render('SolicitudesRutas/Show', compact('solicitudes', 'rutaInfo'));
        return Inertia::render('SolicitudesRutas/Show', [
            'solicitudes' => $solicitudes,
            'rutaInfo' => $rutaInfo,
            'filters' => RequestFacade::only(['estado'])
        ]);
    }



    
    public function aprobar($id)
    {
        $solicitud = SolicitudesRuta::find($id);

        if (!$solicitud) {
            throw new \Exception('Solicitud no encontrada');
        }

        $ruta = Rutas::find($solicitud->id_ruta);
        if (!$ruta) {
            throw new \Exception('Ruta no encontrada');
        }

        if ($ruta->puestos_disponibles <= 0) {
            throw new \Exception('No hay puestos disponibles');
        }

        $estudiante = Estudiantes::find($solicitud->id_estudiante);
        if (!$estudiante) {
            throw new \Exception('Estudiante no encontrado');
        }

        $user = User::find($estudiante->id_user);
        if (!$user) {
            throw new \Exception('Usuario no encontrado');
        }

        $solicitud->estado = 'aprobado';
        $solicitud->save();

        $ruta->puestos_disponibles = $ruta->puestos_disponibles - 1;
        $ruta->save();


        // Enviar correo de aprobación
        Mail::to($user->email)->send(new SolicitudRutasMail($solicitud->estado, $ruta->origen, $ruta->destino, $estudiante->nombre));


        return response()->json([
            'message' => 'Solicitud aprobada exitosamente',
            'solicitud' => $solicitud
        ], 200);
        
    }

    public function rechazar($id)
    {
        $solicitud = SolicitudesRuta::find($id);
        if (!$solicitud) {
            throw new \Exception('Solicitud no encontrada');
        }

        $ruta = Rutas::find($solicitud->id_ruta);
        if (!$ruta) {
            throw new \Exception('Ruta no encontrada');
        }

        $estudiante = Estudiantes::find($solicitud->id_estudiante);
        if (!$estudiante) {
            throw new \Exception('Estudiante no encontrado');
        }


        $user = User::find($estudiante->id_user);
        if (!$user) {
            throw new \Exception('Usuario no encontrado');
        }

        if ($solicitud->estado == 'aprobado') {
            $ruta->puestos_disponibles = $ruta->puestos_disponibles + 1;
            $ruta->save();
        }

        $solicitud->estado = 'rechazado';
        $solicitud->save();



        // Enviar correo de rechazo
        Mail::to($user->email)->send(new SolicitudRutasMail($solicitud->estado, $ruta->origen, $ruta->destino, $estudiante->nombre));

        return response()->json([
            'message' => 'Solicitud rechazada exitosamente',
            'solicitud' => $solicitud
        ], 200);
    }

    public function finalizar($id_ruta)
    {
        try {


            $ruta = Rutas::find($id_ruta);
            if (!$ruta) {
                throw new \Exception('Ruta no encontrada');
            }

            if ($ruta->estado == 0) {

                $solicitudes = SolicitudesRuta::where('id_ruta', $id_ruta)->get();

                foreach ($solicitudes as $solicitud) {


                    $solicitud->estado = 'finalizado';
                    $solicitud->save();

                }

            }



            return back()->with('success', 'Ruta finalizada exitosamente');
        } catch (\Exception $e) {
            Log::error('Error al finalizar la solicitud: ' . $e->getMessage());
            return back()->with('error', 'Error al finalizar la solicitud: ' . $e->getMessage());
        }
    }
}
