<?php

namespace App\Http\Controllers;

use App\Mail\SolicitudConductorMail;
use App\Models\Conductores;
use Illuminate\Support\Facades\Request as RequestFacade;
use App\Models\Estudiantes;
use App\Models\User;
use App\Models\Vehiculos;
use Inertia\Inertia;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Log;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

class SolicitudesConductoresController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    function __construct()
    {
        $this->middleware('permission:solicitudes_conductor-listar|solicitudes_conductor-crear|solicitudes_conductor-editar|solicitudes_conductor-eliminar', ['only' => ['index', 'store']]);
        $this->middleware('permission:solicitudes_conductor-crear', ['only' => ['create', 'store']]);
        $this->middleware('permission:solicitudes_conductor-editar', ['only' => ['edit', 'update']]);
        $this->middleware('permission:solicitudes_conductor-eliminar', ['only' => ['destroy']]);
    }

    public function index(Request $request)
    {
        $solicitudes = Estudiantes::query()
            ->select('id', 'nombre', 'identificacion', 'telefono', 'created_at')
            ->whereHas('conductor', function ($query) {
                $query->where('estado', 0);
            })
            ->when(RequestFacade::input('search'), function ($query, $search) {
                $query->where('nombre', 'like', '%' . $search . '%')
                    ->orWhere('identificacion', 'like', '%' . $search . '%');
            })
            ->orderBy('created_at', 'asc')
            ->paginate(10)
            ->withQueryString();

        return Inertia::render('SolicitudesConductores/Index', [
            'solicitudes' => $solicitudes,
            'filters' => RequestFacade::only(['search'])
        ]);
    }

    public function edit(Request $request, $id_solicitud)
    {
        $solicitud = Estudiantes::with([
            'conductor', 
            'conductor.vehiculos'
        ])->findOrFail($id_solicitud);

        $baseUrl = url('/');
        $url_imagenes = $baseUrl . '/storage/';

        return Inertia::render('SolicitudesConductores/Edit', compact('solicitud', 'url_imagenes'));
    }


    public function update(Request $request, $solicitud)
    {
        $request->validate([
            'respuesta' => 'required',
            'vehiculos' => [
                'required',
                'array',
                function ($attribute, $value, $fail) use ($request) {
                    if ($request->respuesta === 'Aprobar') {
                        $aprobados = collect($value)->filter(fn ($vehiculo) => $vehiculo['estado'] === true);
                        if ($aprobados->isEmpty()) {
                            $fail('Debe haber al menos un vehículo aprobado para aprobar la solicitud.');
                        }
                    }
                },
            ],
        ]);

        if($request->input('respuesta') === 'Aprobar'){
            foreach ($request->vehiculos as $vehiculo) {
                Vehiculos::where('id', $vehiculo['id'])->update(['estado' => $vehiculo['estado']]);
            }

            $solicitudConductor = Conductores::find($solicitud);
            $solicitudConductor->estado = 1;
            $solicitudConductor->save();

            $estudiante = Estudiantes::where('id', $solicitudConductor->id_estudiante)->first();

            if ($estudiante) {
                $usuarioSolicitud = User::where('id', $estudiante->id_user)->first();
                if ($usuarioSolicitud) {
                    $usuarioSolicitud->roles()->attach(6);

                    $estadoCorreo = 'Aprobado';

                    // Enviar correo de aprobación
                    Mail::to($usuarioSolicitud->email)->send(new SolicitudConductorMail($estadoCorreo, $usuarioSolicitud));
                    
                    return Redirect::route('solicitudes-conductores.index')->with('message', 'Registro almacenado.');
                }
            }
        } elseif ($request->input('respuesta') === 'Denegar') {
            // Buscar la solicitud del conductor
            $solicitudConductor = Conductores::find($solicitud);
        
            if ($solicitudConductor) {
                // Eliminar los vehículos asociados al conductor
                Vehiculos::where('id_conductor', $solicitudConductor->id)->delete();
        
                // Obtener el estudiante asociado para acceder a su ID de usuario
                $estudiante = Estudiantes::where('id', $solicitudConductor->id_estudiante)->first();
                
                if ($estudiante) {
                    $estudiante->isConductor = 0;
                    $estudiante->save();

                    $userId = $estudiante->id_user;
                    $rutaCarpeta = "uploads/conductores/{$userId}";

                    if (Storage::disk('public')->exists($rutaCarpeta)) {
                        Storage::disk('public')->deleteDirectory($rutaCarpeta);
                    }

                    $usuarioSolicitud = User::where('id', $estudiante->id_user)->first();

                    $estadoCorreo = 'Rechazado';
                    $comentario = $request->input('comentario');

                    // Enviar correo de aprobación
                    try {
                        Mail::to($usuarioSolicitud->email)->send(new SolicitudConductorMail($estadoCorreo, $usuarioSolicitud, $comentario));
                    } catch (\Exception $e) {
                        Log::error('Error al enviar el correo: ' . $e->getMessage());
                    }
                    
                    // Eliminar el conductor
                    $solicitudConductor->delete();

                    return Redirect::route('solicitudes-conductores.index')->with('message', 'Registro almacenado.');
                }
            }
        }
    }
}
