<?php

namespace App\Http\Controllers;

use App\Models\Conductores;
use App\Models\Estudiantes;
use Illuminate\Support\Facades\Request as RequestFacade;
use App\Models\Gastos;
use App\Models\Rutas;
use App\Models\SolicitudesRuta;
use App\Models\Vehiculos;
use Dotenv\Exception\ValidationException;
use Inertia\Inertia;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Carbon;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SolicitarViajeController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    function __construct()
    {
        $this->middleware('permission:solicitar_viajes-listar|solicitar_viajes-crear|solicitar_viajes-editar|solicitar_viajes-eliminar', ['only' => ['index', 'store']]);
        $this->middleware('permission:solicitar_viajes-crear', ['only' => ['create', 'store']]);
        $this->middleware('permission:solicitar_viajes-editar', ['only' => ['edit', 'update']]);
        $this->middleware('permission:solicitar_viajes-eliminar', ['only' => ['destroy']]);
    }

    public function index(Request $request)
    {
        // Obtener el usuario logueado
        $userAuth = Auth::user();

        // Obtener el estudiante del usuario
        $estudiante = Estudiantes::select('id', 'nombre')->where('id_user', $userAuth->id)->first();

        // Obtener el ID del estudiante
        $estudianteId = $estudiante ? $estudiante->id : null;

        // Obtener el conductor si el usuario es un conductor
        $conductor = $estudiante ? Conductores::select('id', 'licencia')->where('id_estudiante', $estudianteId)->first() : null;
        $conductorId = $conductor ? $conductor->id : null;

        // Obtener los ID de rutas que el estudiante ya solicitó
        $rutasSolicitadas = $estudianteId ? 
            SolicitudesRuta::where('id_estudiante', $estudianteId)->pluck('id_ruta')->toArray() : [];

        // Obtener filtros desde la petición
        $origen = $request->input('origen');
        $destino = $request->input('destino');
        $fecha = $request->input('fecha_hora');

        $rutas = Rutas::query()
            ->where('estado', 0)
            ->where('fecha_hora', '>=', Carbon::now())
            ->where('puestos_disponibles', '>=', 1)
            ->with(['vehiculo:id,marca_vehiculo,linea_vehiculo,placa'])
            ->when($origen, function ($query, $origen) {
                return $query->where('origen', 'like', "%$origen%");
            })
            ->when($destino, function ($query, $destino) {
                return $query->where('destino', 'like', "%$destino%");
            })
            ->when($fecha, function ($query, $fecha) {
                $fechaInicio = Carbon::parse($fecha)->startOfMinute();
                $fechaFin = $fechaInicio->copy()->addHour();
            
                return $query->whereBetween('fecha_hora', [$fechaInicio, $fechaFin]);
            })
            ->when($conductorId, function ($query) use ($conductorId) {
                // Si el usuario es conductor, excluir sus propias rutas
                return $query->where('id_conductor', '!=', $conductorId);
            })
            ->when(!empty($rutasSolicitadas), function ($query) use ($rutasSolicitadas) {
                // Excluir rutas que el estudiante ya ha solicitado
                return $query->whereNotIn('id', $rutasSolicitadas);
            })
            ->orderBy('fecha_hora', 'asc')
            ->paginate(5)
            ->withQueryString();

        return Inertia::render('SolicitarViaje/Index', [
            'rutas' => $rutas,
            'filters' => $request->only(['origen', 'destino', 'fecha_hora'])
        ]);
    }



    public function show($id)
    {
        $ruta = Rutas::with([
            'conductor:id,id_estudiante,puntaje',
            'conductor.estudiante:id,nombre,foto_perfil',
            'vehiculo:id,placa,linea_vehiculo,marca_vehiculo,foto_vehiculo'
        ])->findOrFail($id);
    
        $api_key = env('API_RUTE_KEY');
        $api_route = env('API_RUTE');

        $baseUrl = url('/');
        $url_imagenes = $baseUrl . '/storage/';
    
        return Inertia::render('SolicitarViaje/Add', [
            'ruta' => $ruta,
            'api_key' => $api_key,
            'api_route' => $api_route,
            'url_imagenes' => $url_imagenes
        ]);
    }
    


    public function store(Request $request)
    {
        
        $userAuth = Auth::user();

        $estudiante = Estudiantes::select('id', 'nombre')->where('id_user', $userAuth->id)->first();

        $request->validate([
            'id_ruta' => 'required',
            'observacion' => 'nullable|string|max:255',
        ]);

        $ruta = SolicitudesRuta::create([
            'id_ruta' =>  $request->id_ruta,
            'id_estudiante' => $estudiante->id,
            'observaciones' => $request->observacion,
            'estado' => 'pendiente',
        ]);

    }
}
