<?php

namespace App\Http\Controllers;

use App\Mail\CancelacionRutasMail;
use App\Models\Calificaciones;
use App\Models\Conductores;
use App\Models\Estudiantes;
use Illuminate\Support\Facades\Request as RequestFacade;
use App\Models\Rutas;
use App\Models\SolicitudesRuta;
use App\Models\Vehiculos;
use Inertia\Inertia;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;

class RutasController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    function __construct()
    {
        $this->middleware('permission:rutas-listar|rutas-crear|rutas-editar|rutas-eliminar', ['only' => ['index', 'store']]);
        $this->middleware('permission:rutas-crear', ['only' => ['create', 'store']]);
        $this->middleware('permission:rutas-editar', ['only' => ['edit', 'update']]);
        $this->middleware('permission:rutas-eliminar', ['only' => ['destroy']]);
    }

    public function index(Request $request)
    {
        $userAuth = Auth::user();

        $estudiante = Estudiantes::select('id', 'nombre')->where('id_user', $userAuth->id)->first();
        $conductor = Conductores::select('id', 'licencia')->where('id_estudiante', $estudiante->id)->first();

        $estado = $request->input('estado', 0);

        $rutas = Rutas::query()
            ->select('id', 'origen', 'destino', 'fecha_hora', 'puestos_disponibles', 'estado', 'id_vehiculo', 'id_conductor')
            ->where('id_conductor', $conductor->id)
            ->where('estado', $estado)
            ->with(['vehiculo:id,marca_vehiculo,linea_vehiculo,placa'])
            ->orderBy('created_at', 'desc')
            ->paginate(10)
            ->withQueryString();

        // Modificar cada ruta agregando campo personalizado
        $rutas->getCollection()->transform(function ($ruta) {
            $ruta->loadCount('calificacionesPendientes');
            $ruta->pendiente_calificacion = $ruta->calificaciones_pendientes_count > 0;
            return $ruta;
        });

        return Inertia::render('Rutas/Index', [
            'rutas' => $rutas,
            'filters' => RequestFacade::only(['estado'])
        ]);
    }


    public function create()
    {
        $userAuth = Auth::user();

        $estudiante = Estudiantes::select('id', 'nombre')->where('id_user', $userAuth->id)->first();

        $conductor = Conductores::select('id', 'licencia')->where('id_estudiante', $estudiante->id)->first();

        $vehiculos = Vehiculos::select('id', 'placa', 'marca_vehiculo')->where('id_conductor', $conductor->id)->get();

        $api_key = env('API_RUTE_KEY');
        $api_route = env('API_RUTE');
      
        return Inertia::render('Rutas/Add', compact('vehiculos', 'conductor', 'api_key', 'api_route'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'origen' => 'required|string|max:500',
            'destino' => 'required|string|max:500',
            'lat_origen' => 'required',
            'long_origen' => 'required',
            'lat_destino' => 'required',
            'long_destino' => 'required',
            'vehiculo' => 'required',
            'puestos_disponibles' => 'required',
            'fecha_hora' => 'required',
            'conductor' => 'required',
            'kilometros' => 'required',
            'indicaciones' => 'nullable|string|max:255',
        ]);

        $ruta = Rutas::create([
            'id_conductor' => $request->conductor,
            'id_vehiculo' => $request->vehiculo,
            'origen' => $request->origen,
            'destino' => $request->destino,
            'fecha_hora' => $request->fecha_hora,
            'puestos_disponibles' => $request->puestos_disponibles,
            'lat_origen' => $request->lat_origen,
            'long_origen' => $request->long_origen,
            'lat_destino' => $request->lat_destino,
            'long_destino' => $request->long_destino,
            'kilometros' => $request->kilometros,
            'indicaciones' => $request->indicaciones,
            'estado' => 0,
        ]);

        return to_route('rutas.index');
    }


    public function edit(Request $request)
    {

    }

    public function update(Request $request, $id)
    {
        
        $ruta = Rutas::findOrFail($id);

        if ($request->estado === 1) {

            $solicitudes = SolicitudesRuta::where('id_ruta', $id)
                ->where('estado', 'aprobado')
                ->get();
    
            $conductor = Conductores::where('id', $ruta->id_conductor)->first();
            $id_estudiante_conductor = $conductor->id_estudiante; // <- Aquí obtienes el estudiante del conductor
    
            foreach ($solicitudes as $solicitud) {
                // Calificación del pasajero
                Calificaciones::create([
                    'id_evaluador' => $id_estudiante_conductor,
                    'id_evaluado' => $solicitud->id_estudiante,
                    'id_ruta' => $id,
                    'puntaje' => null,
                    'observaciones' => null,
                    'tipo_evaluado' => 'Pasajero',
                    'estado' => 0,
                ]);
    
                // Calificación del conductor
                Calificaciones::create([
                    'id_evaluador' => $solicitud->id_estudiante,
                    'id_evaluado' => $id_estudiante_conductor, // <- Aquí también usas el estudiante del conductor
                    'id_ruta' => $id,
                    'puntaje' => null,
                    'observaciones' => null,
                    'tipo_evaluado' => 'Conductor',
                    'estado' => 0,
                ]);
            }
    
            if ($solicitudes->isNotEmpty()) {
                SolicitudesRuta::where('id_ruta', $id)
                    ->where('estado', 'aprobado')
                    ->update(['estado' => 'finalizado']);
            }
    
            $ruta->estado = $request->estado;
            $ruta->save();
        }else if($request->estado === 2){

            $solicitudes = SolicitudesRuta::where('id_ruta', $id)
            ->where('estado', 'aprobado') // Solo solicitudes aprobadas
            ->with('estudiante.user') // Cargar relación del estudiante y usuario
            ->get();

            // Actualizar estado de las solicitudes a "Cancelado"
            SolicitudesRuta::where('id_ruta', $id)
                ->where('estado', 'aprobado')
                ->update(['estado' => 'cancelado']);

            // Enviar correo a cada estudiante con solicitud aprobada
            foreach ($solicitudes as $solicitud) {
                $estudiante = $solicitud->estudiante;
                $usuario = $estudiante->user ?? null; // Obtener el usuario del estudiante

                if ($usuario && $usuario->email) { // Verificar si el usuario tiene email
                    Mail::to($usuario->email)->send(new CancelacionRutasMail(
                        'Cancelado',
                        $ruta->origen,
                        $ruta->destino,
                        $estudiante->nombre
                    ));
                }
            }

            // Actualizar el estado de la ruta
            $ruta->estado = $request->estado;
            $ruta->save();
        }

        $conductor = Conductores::find($ruta->id_conductor);
        $id_estudiante_conductor = $conductor->id_estudiante;
    
        return response()->json([
            "message" => "Estado actualizado correctamente",
            "estado" => $ruta->estado,
            "calificaciones" => Calificaciones::where('id_ruta', $id)
                ->where('id_evaluador', $id_estudiante_conductor)
                ->where('tipo_evaluado', 'Pasajero')
                ->where('estado', 0)
                ->with(['estudiante:id,nombre,foto_perfil'])
                ->get()
        ], 200);
    }
}
