<?php

namespace App\Http\Controllers;

use App\Mail\SolicitudConductorMail;
use App\Models\Conductores;
use Illuminate\Support\Facades\Request as RequestFacade;
use App\Models\Estudiantes;
use App\Models\Rutas;
use App\Models\User;
use App\Models\Vehiculos;
use Carbon\Carbon;
use Inertia\Inertia;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Log;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

class DashboardController extends Controller
{

    public function resumen(Request $request)
    {
        $ahora = Carbon::now();
        $haceTreintaMinutos = $ahora->copy()->subMinutes(30);

        $viajesEnCurso = Rutas::whereBetween('fecha_hora', [$haceTreintaMinutos, $ahora])
        ->whereNotIn('estado', [1, 2])
        ->count();
        
        $rutasProgramadas = Rutas::where('fecha_hora', '>', $ahora)
            ->where('estado', 0)
            ->count();
        
        $usuariosActivos = User::where('estado', 'Activo')
            ->count();

        $viajesCompletados = Rutas::where('estado', 1)
            ->where('fecha_hora', '<=', $ahora)
            ->count();

        return response()->json([
            'viajes_en_curso' => $viajesEnCurso,
            'rutas_programadas' => $rutasProgramadas,
            'usuarios_activos' => $usuariosActivos,
            'viajes_completados' => $viajesCompletados,
        ]);
    }

    public function rutas(Request $request)
    {
        $perPage = $request->input('per_page', 10);

        $rutas = Rutas::select('id', 'origen', 'destino', 'fecha_hora', 'estado', 'id_conductor', 'id_vehiculo', 'created_at')
            ->with([
                'conductor:id,id_estudiante',
                'conductor.estudiante:id,nombre',
                'vehiculo:id,marca_vehiculo,linea_vehiculo'
            ])
            ->orderBy('created_at', 'desc')
            ->paginate($perPage);

        return response()->json($rutas);
    }

    public function conductores(Request $request)
    {
        $perPage = $request->input('per_page', 10);

        $conductores = Conductores::select('id', 'puntaje', 'estado', 'id_estudiante')
            ->with([
                'estudiante:id,nombre,identificacion,telefono',
            ])
            ->where('estado', 1)
            ->orderBy('created_at', 'desc')
            ->paginate($perPage);

        return response()->json($conductores);
    }

    public function getDashboardMetrics()
    {
        $rutas = Rutas::where('estado', 1)->get();

        $total_km = $rutas->sum('kilometros');

        $autos_eliminados = round($total_km / 15, 2);
        $arboles_equivalentes = round($total_km / 100, 2);
        $ahorro_gasolina = round($total_km * 0.08, 2);
        $co2_total = round($total_km * 0.21, 2); // Estimación total de CO₂

        // === CO₂ ahorrado por mes ===
        $co2_por_mes = $rutas->groupBy(function ($ruta) {
            return Carbon::parse($ruta->fecha_hora)->format('n'); // 1 (enero) a 12 (diciembre)
        })->map(function ($group) {
            return round($group->sum('kilometros') * 0.21, 2);
        });

        // Inicializar todos los meses con 0
        $co2_mensual = [];
        for ($i = 1; $i <= 12; $i++) {
            $co2_mensual[] = $co2_por_mes->get($i, 0);
        }

        return response()->json([
            'autos_eliminados' => $autos_eliminados,
            'arboles_equivalentes' => $arboles_equivalentes,
            'ahorro_gasolina' => $ahorro_gasolina,
            'kilometros_compartidos' => $total_km,
            'co2_mensual' => $co2_mensual,
            'co2_total' => $co2_total,
        ]);
    }


    public function update(Request $request, $solicitud)
    {
        $request->validate([
            'respuesta' => 'required',
            'vehiculos' => [
                'required',
                'array',
                function ($attribute, $value, $fail) use ($request) {
                    if ($request->respuesta === 'Aprobar') {
                        $aprobados = collect($value)->filter(fn ($vehiculo) => $vehiculo['estado'] === true);
                        if ($aprobados->isEmpty()) {
                            $fail('Debe haber al menos un vehículo aprobado para aprobar la solicitud.');
                        }
                    }
                },
            ],
        ]);

        if($request->input('respuesta') === 'Aprobar'){
            foreach ($request->vehiculos as $vehiculo) {
                Vehiculos::where('id', $vehiculo['id'])->update(['estado' => $vehiculo['estado']]);
            }

            $solicitudConductor = Conductores::find($solicitud);
            $solicitudConductor->estado = 1;
            $solicitudConductor->save();

            $estudiante = Estudiantes::where('id', $solicitudConductor->id_estudiante)->first();

            if ($estudiante) {
                $usuarioSolicitud = User::where('id', $estudiante->id_user)->first();
                if ($usuarioSolicitud) {
                    $usuarioSolicitud->roles()->attach(6);

                    $estadoCorreo = 'Aprobado';

                    // Enviar correo de aprobación
                    Mail::to($usuarioSolicitud->email)->send(new SolicitudConductorMail($estadoCorreo, $usuarioSolicitud));
                    
                    return Redirect::route('solicitudes-conductores.index')->with('message', 'Registro almacenado.');
                }
            }
        } elseif ($request->input('respuesta') === 'Denegar') {
            // Buscar la solicitud del conductor
            $solicitudConductor = Conductores::find($solicitud);
        
            if ($solicitudConductor) {
                // Eliminar los vehículos asociados al conductor
                Vehiculos::where('id_conductor', $solicitudConductor->id)->delete();
        
                // Obtener el estudiante asociado para acceder a su ID de usuario
                $estudiante = Estudiantes::where('id', $solicitudConductor->id_estudiante)->first();
                
                if ($estudiante) {
                    $estudiante->isConductor = 0;
                    $estudiante->save();

                    $userId = $estudiante->id_user;
                    $rutaCarpeta = "uploads/conductores/{$userId}";

                    if (Storage::disk('public')->exists($rutaCarpeta)) {
                        Storage::disk('public')->deleteDirectory($rutaCarpeta);
                    }

                    $usuarioSolicitud = User::where('id', $estudiante->id_user)->first();

                    $estadoCorreo = 'Rechazado';
                    $comentario = $request->input('comentario');

                    // Enviar correo de aprobación
                    try {
                        Mail::to($usuarioSolicitud->email)->send(new SolicitudConductorMail($estadoCorreo, $usuarioSolicitud, $comentario));
                    } catch (\Exception $e) {
                        Log::error('Error al enviar el correo: ' . $e->getMessage());
                    }
                    
                    // Eliminar el conductor
                    $solicitudConductor->delete();

                    return Redirect::route('solicitudes-conductores.index')->with('message', 'Registro almacenado.');
                }
            }
        }
    }
}
