<?php

namespace App\Http\Controllers;

use App\Models\Calificaciones;
use App\Models\Conductores;
use App\Models\Estudiantes;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class CalificacionesController extends Controller
{
    public function edit(Request $request, $id){
        
        $conductor = Conductores::find($request->id_conductor);
        $id_estudiante_conductor = $conductor->id_estudiante;

        $calificaciones = Calificaciones::where('id_ruta', $id)
        ->where('id_evaluador', $id_estudiante_conductor)
        ->where('tipo_evaluado', 'Pasajero')
        ->where('estado', 0)
        ->with(['estudiante:id,nombre,foto_perfil'])
        ->get();

        return response()->json($calificaciones);
    }
    
    public function store(Request $request)
    {
        $request->validate([
            'calificaciones' => 'required|array',
            'calificaciones.*.id' => 'required|integer|exists:calificaciones,id',
            'calificaciones.*.id_evaluado' => 'required|integer|exists:estudiante,id',
            'calificaciones.*.puntaje' => 'required|numeric|min:1|max:5',
        ]);

        DB::beginTransaction();
        try {
            foreach ($request->calificaciones as $item) {
                // Buscar el registro existente y actualizar
                $calificacion = Calificaciones::findOrFail($item['id']);

                $calificacion->update([
                    'puntaje'        => $item['puntaje'],
                    'observaciones'  => $item['observaciones'] ?? null,
                    'estado'         => 1,
                ]);

                // Calcular nuevo promedio del estudiante
                $promedio = Calificaciones::where('id_evaluado', $item['id_evaluado'])->avg('puntaje');

                // Actualizar el puntaje del estudiante
                $estudiante = Estudiantes::find($item['id_evaluado']);
                $estudiante->puntaje = round($promedio, 2);
                $estudiante->save();
            }

            DB::commit();

            return response()->json(['message' => 'Calificaciones actualizadas correctamente'], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Error al actualizar calificaciones',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function getCalificarConductores()
    {
        $user = auth()->user();

        $estudiante = Estudiantes::where('id_user', $user->id)->first();

        if (!$estudiante) {
            return response()->json(['message' => 'Estudiante no encontrado'], 404);
        }

        $calificaciones = Calificaciones::where('id_evaluador', $estudiante->id)
            ->where('tipo_evaluado', 'Conductor')
            ->where('estado', 0)
            ->with(['estudiante:id,nombre,foto_perfil', 'ruta:id,origen,destino'])
            ->get();

        return response()->json($calificaciones);
    }

    public function updateCalificacionConductor(Request $request)
    {
        $request->validate([
            'calificaciones' => 'required|array',
            'calificaciones.*.id' => 'required|integer|exists:calificaciones,id',
            'calificaciones.*.id_evaluado' => 'required|integer|exists:estudiante,id',
            'calificaciones.*.puntaje' => 'required|numeric|min:1|max:5',
        ]);

        DB::beginTransaction();
        try {
            foreach ($request->calificaciones as $item) {
                // Actualizar la calificación
                $calificacion = Calificaciones::findOrFail($item['id']);
                $calificacion->update([
                    'puntaje'        => $item['puntaje'],
                    'observaciones'  => $item['observaciones'] ?? null,
                    'estado'         => 1,
                ]);

                // Obtener promedio del evaluado
                $promedio = Calificaciones::where('id_evaluado', $item['id_evaluado'])->avg('puntaje');

                // Buscar el conductor relacionado con ese estudiante
                $conductor = Conductores::where('id_estudiante', $item['id_evaluado'])->first();

                if ($conductor) {
                    $conductor->puntaje = round($promedio, 2);
                    $conductor->save();
                }
            }

            DB::commit();
            return response()->json(['message' => 'Calificaciones actualizadas correctamente'], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Error al actualizar calificaciones',
                'error' => $e->getMessage()
            ], 500);
        }
    }

}
